<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Mail;
use App\Mail\GarageGateRequestMail;
use App\Mail\SendMessageToEndUser;

class garageGateFormController extends Controller
{
    public function showForm()
    {
        return view('garazskapu');
    }

    public function submitForm(Request $request)
    {
        //dd($request);
        $request->merge([
            // checkboxok: ha nincs benne a request-ben, legyen 0
            'window' => $request->has('window') ? '1' : '0',
            'personalGate' => $request->has('personalGate') ? '1' : '0',
            'installation' => $request->has('installation') ? '1' : '0',
            // további checkboxok hasonlóan...
        ]);

        $rules = [
            "color" => "bail|required|string",
            "sizeWidth" => "bail|required|integer",
            "sizeHeight" => "bail|required|integer",
            "controlByPhone" => "bail|nullable",
            "window" => "bail|nullable",
            "personalGate" => "bail|nullable",
            "installation" => "bail|nullable",
            "name" => "bail|required|string",
            "email" => "bail|required|email|string",
            "phone" => "bail|required|string"
        ];

        $messages = [
            // szín
            'color.required' => 'A szín megadása kötelező.',

            // méretek
            'sizeWidth.required' => 'A szélesség megadása kötelező.',
            'sizeWidth.numeric' => 'A szélesség szám kell legyen.',
            'sizeWidth.min' => 'A szélesség legalább :min kell legyen.',

            'sizeHeight.required' => 'A magasság megadása kötelező.',
            'sizeHeight.numeric' => 'A magasság szám kell legyen.',
            'sizeHeight.min' => 'A magasság legalább :min kell legyen.',

            'sill.required_if' => 'Kérjük adja meg a küszöb típusát, ha a személykapu opció be van kapcsolva.',
            'sill.integer' => 'A küszöb értékének egész számnak kell lennie.',
            'sill.in' => 'A kiválasztott küszöb típus érvénytelen.',

            // photocell típus
            'photoCellType.required' => 'Kérjük válassza ki a fotocella típusát.',
            'photoCellType.in' => 'Érvénytelen fotocella típus.',

            // installationPlace
            'installationPlace.required' => 'Kérjük adja meg a beszerelés helyét.',
            'installationPlace.string' => 'A beszerelés helye szöveg legyen.',
            'installationPlace.max' => 'A beszerelés helye maximum :max karakter lehet.',

            // kapcsolattartó
            'name.required' => 'A név megadása kötelező.',
            'name.string' => 'A név szöveg kell legyen.',
            'name.max' => 'A név maximum :max karakter lehet.',

            'email.required' => 'Az e-mail megadása kötelező.',
            'email.email' => 'Érvényes e-mail címet adjon meg.',
            'email.max' => 'Az e-mail maximum :max karakter lehet.',

            'phone.required' => 'A telefonszám megadása kötelező.',
            'phone.string' => 'A telefonszám szövegként adandó meg.',
            'phone.max' => 'A telefonszám maximum :max karakter lehet.',

            // windowsCount példájához (ha használod)
            'windowsCount.required_if' => 'Add meg, hány ablak van, ha az ablak opció be van kapcsolva.',
            'windowsCount.integer' => 'Az ablakok számának egész számnak kell lennie.',

            // window frame
            'frameColor.required_if' => 'Válassza ki a keret színét, ha az ablak opció be van kapcsolva.',
            'frameColor.integer' => 'A keret színe számként (azonosítóként) kell megadni.',
            'frameColor.in' => 'A kiválasztott keret színe érvénytelen.',

            // általános fallback üzenetek (opcionális)
            'required' => 'A(z) :attribute mező kitöltése kötelező.',
            'numeric' => 'A(z) :attribute mezőnek számnak kell lennie.',
            'integer' => 'A(z) :attribute mezőnek egész számnak kell lennie.',
            'in' => 'A kiválasztott érték a(z) :attribute mezőhöz érvénytelen.',
            'max' => 'A(z) :attribute mező maximum :max karakter lehet.',
            'min' => 'A(z) :attribute mező minimális értéke :min.',
        ];

        $validator = Validator::make($request->all(), $rules, $messages);

        $validator->sometimes("windowsCount", "integer", function ($input) {
            return isset($input->window) && (string)$input->window === '1';
        });

        $validator->sometimes("frameColor", "integer|in:1,2", function ($input) {
            return isset($input->window) && (string)$input->window === '1';
        });

        $validator->sometimes("sill", "integer|in:1,2", function ($input) {
            return isset($input->personalGate) && (string)$input->personalGate === '1';
        });

        $validator->sometimes("installationPlace", "string", function ($input) {
            return isset($input->installation) && (string)$input->installation === '1';
        });

        if ($validator->fails()) {
            // debug: dd($validator->errors()->toArray());
            return back()->withErrors($validator)->withInput();
        }

        //levél küldés
        $data = $validator->validated();

        // Send to admin
        Mail::to('info@domakaputech.hu')->send(new GarageGateRequestMail($data));

        $name = $request->name;
        $senderMessage = "Köszönjük az üzenetét, hamarosan felvesszük Önnel a kapcsolatot.";
        $email = $request->email;
        $mailData = [
            'url' =>  config('app.url'),
        ];
        
        try {
            Mail::to($email)->send(new SendMessageToEndUser($name, $senderMessage, $mailData));
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', 'Az üzenetet nem sikerült elküldenünk. Kérjük próbálja újra később!');
        }

        return back()->with('success', 'Köszönjük az üzenetét, hamarosan felvesszük Önnel a kapcsolatot.!');
    }
}
